<?php
/**
 * Manifest - Zentrale Metadaten-Verwaltung für JBM-Backups
 * 
 * Das Manifest enthält alle Informationen über ein Backup:
 * - Metadaten (WordPress-Version, Site-URL, Backup-Typ, Zeitstempel)
 * - Dateiliste mit Checksums
 * - Segment-Mapping (welche Datei in welchem Segment)
 * - Plugin-Inventar
 * - Restore-Anweisungen
 * 
 * @package JenvaBackupMigration
 * @since 2.0.0
 */

namespace JenvaBackupMigration\Core;

if (!defined('ABSPATH')) {
    exit;
}

class Manifest {
    
    /** @var string Manifest-Version für Kompatibilität */
    const VERSION = '2.0.0';
    
    /** @var array Manifest-Daten */
    private $data = [];
    
    /** @var string Pfad zur Manifest-Datei */
    private $path;
    
    /**
     * Erstellt ein neues Manifest
     * 
     * @param string $backup_type 'full', 'database', 'files', 'slim'
     */
    public function __construct(string $backup_type = 'full') {
        $this->data = [
            'manifest_version' => self::VERSION,
            'backup_type' => $backup_type,
            'created_at' => gmdate('Y-m-d\TH:i:s\Z'),
            'created_timestamp' => time(),
            
            // WordPress-Metadaten
            'wordpress' => [
                'version' => get_bloginfo('version'),
                'site_url' => get_site_url(),
                'home_url' => get_home_url(),
                'db_prefix' => $GLOBALS['wpdb']->prefix,
                'charset' => get_bloginfo('charset'),
                'language' => get_locale(),
                'multisite' => is_multisite(),
            ],
            
            // PHP/Server-Info
            'environment' => [
                'php_version' => PHP_VERSION,
                'mysql_version' => $this->get_mysql_version(),
                'server_software' => $_SERVER['SERVER_SOFTWARE'] ?? 'unknown',
            ],
            
            // Theme-Info
            'theme' => [
                'name' => get_stylesheet(),
                'parent' => get_template(),
                'version' => wp_get_theme()->get('Version'),
            ],
            
            // Segmente (werden beim Backup gefüllt)
            'segments' => [],
            
            // Dateiliste mit Checksums
            'files' => [],
            
            // Plugin-Inventar
            'plugins' => [],
            
            // Datenbank-Info
            'database' => [
                'tables' => [],
                'total_rows' => 0,
                'cleaned' => false,
                'cleanup_stats' => [],
            ],
            
            // Statistiken
            'stats' => [
                'total_files' => 0,
                'total_size' => 0,
                'compressed_size' => 0,
                'excluded_files' => 0,
                'excluded_size' => 0,
            ],
            
            // Checksums für Integrität
            'checksums' => [
                'algorithm' => 'sha256',
                'manifest' => null, // Wird beim Finalisieren berechnet
            ],
        ];
    }
    
    /**
     * Lädt ein bestehendes Manifest
     * 
     * @param string $path Pfad zur manifest.json
     * @return self
     * @throws \Exception Wenn Manifest ungültig
     */
    public static function load(string $path): self {
        if (!file_exists($path)) {
            throw new \Exception("Manifest nicht gefunden: $path");
        }
        
        $json = file_get_contents($path);
        $data = json_decode($json, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new \Exception("Ungültiges Manifest-Format: " . json_last_error_msg());
        }
        
        // Version prüfen
        if (!isset($data['manifest_version'])) {
            throw new \Exception("Manifest-Version fehlt");
        }
        
        $manifest = new self($data['backup_type'] ?? 'full');
        $manifest->data = $data;
        $manifest->path = $path;
        
        return $manifest;
    }
    
    /**
     * Speichert das Manifest
     * 
     * @param string $path Zielpfad
     * @return bool
     */
    public function save(string $path): bool {
        $this->path = $path;
        
        // Finale Checksumme berechnen
        $data_for_checksum = $this->data;
        $data_for_checksum['checksums']['manifest'] = null;
        $this->data['checksums']['manifest'] = hash('sha256', json_encode($data_for_checksum));
        
        $json = json_encode($this->data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
        
        return file_put_contents($path, $json) !== false;
    }
    
    /**
     * Fügt ein Segment hinzu
     * 
     * @param string $name Segment-Name (z.B. 'uploads-001')
     * @param string $type Typ ('database', 'uploads', 'themes', 'plugins')
     * @param int $size Größe in Bytes
     * @param string $checksum SHA256 des Segments
     */
    public function addSegment(string $name, string $type, int $size, string $checksum): void {
        $this->data['segments'][$name] = [
            'type' => $type,
            'size' => $size,
            'checksum' => $checksum,
            'created_at' => gmdate('Y-m-d\TH:i:s\Z'),
        ];
    }
    
    /**
     * Fügt eine Datei zum Manifest hinzu
     * 
     * @param string $relative_path Relativer Pfad
     * @param int $size Größe in Bytes
     * @param string $checksum SHA256
     * @param string $segment In welchem Segment
     */
    public function addFile(string $relative_path, int $size, string $checksum, string $segment): void {
        $this->data['files'][$relative_path] = [
            'size' => $size,
            'checksum' => $checksum,
            'segment' => $segment,
        ];
        
        $this->data['stats']['total_files']++;
        $this->data['stats']['total_size'] += $size;
    }
    
    /**
     * Fügt Plugin zum Inventar hinzu
     * 
     * @param array $plugin_data Plugin-Informationen
     */
    public function addPlugin(array $plugin_data): void {
        $this->data['plugins'][] = [
            'slug' => $plugin_data['slug'],
            'name' => $plugin_data['name'],
            'version' => $plugin_data['version'],
            'active' => $plugin_data['active'],
            'source' => $plugin_data['source'], // 'wordpress.org', 'premium', 'custom'
            'update_url' => $plugin_data['update_url'] ?? null,
            'backed_up' => $plugin_data['backed_up'] ?? false,
        ];
    }
    
    /**
     * Setzt Datenbank-Informationen
     * 
     * @param array $tables Tabellen-Info
     * @param array $cleanup_stats Bereinigungsstatistiken
     */
    public function setDatabaseInfo(array $tables, array $cleanup_stats = []): void {
        $total_rows = 0;
        foreach ($tables as $table) {
            $total_rows += $table['rows'] ?? 0;
        }
        
        $this->data['database'] = [
            'tables' => $tables,
            'total_rows' => $total_rows,
            'cleaned' => !empty($cleanup_stats),
            'cleanup_stats' => $cleanup_stats,
        ];
    }
    
    /**
     * Markiert eine Datei als ausgeschlossen
     * 
     * @param int $size Größe der ausgeschlossenen Datei
     */
    public function addExcluded(int $size): void {
        $this->data['stats']['excluded_files']++;
        $this->data['stats']['excluded_size'] += $size;
    }
    
    /**
     * Setzt die komprimierte Gesamtgröße
     * 
     * @param int $size Größe in Bytes
     */
    public function setCompressedSize(int $size): void {
        $this->data['stats']['compressed_size'] = $size;
    }
    
    /**
     * Getter für Manifest-Daten
     */
    public function get(string $key, $default = null) {
        return $this->data[$key] ?? $default;
    }
    
    /**
     * Gibt das komplette Manifest als Array zurück
     */
    public function toArray(): array {
        return $this->data;
    }
    
    /**
     * Prüft Manifest-Integrität
     * 
     * @return bool
     */
    public function verify(): bool {
        $stored_checksum = $this->data['checksums']['manifest'] ?? null;
        if (!$stored_checksum) {
            return false;
        }
        
        $data_for_checksum = $this->data;
        $data_for_checksum['checksums']['manifest'] = null;
        $calculated = hash('sha256', json_encode($data_for_checksum));
        
        return hash_equals($stored_checksum, $calculated);
    }
    
    /**
     * Gibt die WordPress Site-URL aus dem Manifest zurück
     */
    public function getSiteUrl(): string {
        return $this->data['wordpress']['site_url'] ?? '';
    }
    
    /**
     * Gibt den DB-Prefix aus dem Manifest zurück
     * 
     * WICHTIG: Falls kein Prefix im Manifest vorhanden ist (alte Backups),
     * verwenden wir das aktuelle Prefix, NICHT hardcoded 'wp_'!
     */
    public function getDbPrefix(): string {
        global $wpdb;
        return $this->data['wordpress']['db_prefix'] ?? $wpdb->prefix;
    }
    
    /**
     * Gibt die Plugin-Liste zurück
     */
    public function getPlugins(): array {
        return $this->data['plugins'] ?? [];
    }
    
    /**
     * Gibt die Segment-Liste zurück
     */
    public function getSegments(): array {
        return $this->data['segments'] ?? [];
    }
    
    /**
     * Hilfsfunktion: MySQL-Version ermitteln
     */
    private function get_mysql_version(): string {
        global $wpdb;
        return $wpdb->get_var("SELECT VERSION()") ?? 'unknown';
    }
}

